// QuickCopy - Developer Clipboard Tools
// Copy utilities for developers

// Premium features check (simple version - can integrate ExtensionPay later)
let isPremium = false;

// Check premium status from storage
chrome.storage.local.get(['isPremium'], (result) => {
  isPremium = result.isPremium || false;
  updatePremiumUI();
});

function updatePremiumUI() {
  const premiumBtns = document.querySelectorAll('.premium-btn');
  premiumBtns.forEach(btn => {
    if (isPremium) {
      btn.classList.remove('locked');
      const lockIcon = btn.querySelector('.lock-icon');
      if (lockIcon) lockIcon.remove();
    }
  });
}

async function copyToClipboard(text, buttonId) {
  try {
    await navigator.clipboard.writeText(text);
    showSuccess(buttonId);
    // Save to history
    saveToHistory(text);
  } catch (err) {
    console.error('Failed to copy:', err);
  }
}

function showSuccess(buttonId) {
  const btn = document.getElementById(buttonId);
  btn.classList.add('success');
  setTimeout(() => {
    btn.classList.remove('success');
  }, 1500);
}

async function getCurrentTab() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  return tab;
}

// Save copy to history (premium feature)
function saveToHistory(text) {
  if (!isPremium) return;
  chrome.storage.local.get(['copyHistory'], (result) => {
    const history = result.copyHistory || [];
    history.unshift({ text, timestamp: Date.now() });
    // Keep only last 50 items
    const trimmed = history.slice(0, 50);
    chrome.storage.local.set({ copyHistory: trimmed });
  });
}

// Show upgrade prompt for premium features
function showUpgradePrompt() {
  const existingPrompt = document.getElementById('upgrade-prompt');
  if (existingPrompt) return;

  const prompt = document.createElement('div');
  prompt.id = 'upgrade-prompt';
  prompt.innerHTML = `
    <div class="upgrade-overlay">
      <div class="upgrade-modal">
        <h3>🔓 Unlock Premium</h3>
        <p>Get access to all features:</p>
        <ul>
          <li>✓ Copy all open tabs</li>
          <li>✓ Custom templates</li>
          <li>✓ Copy history</li>
          <li>✓ Keyboard shortcuts</li>
        </ul>
        <a href="https://primedirectiveshop.danprice.ai/quickcopy/upgrade" target="_blank" class="upgrade-btn">Upgrade for $3</a>
        <button class="close-btn" onclick="this.parentElement.parentElement.remove()">Maybe later</button>
      </div>
    </div>
  `;
  document.body.appendChild(prompt);
}

// === FREE FEATURES ===

// Copy URL
document.getElementById('copyUrl').addEventListener('click', async () => {
  const tab = await getCurrentTab();
  copyToClipboard(tab.url, 'copyUrl');
});

// Copy Page Title
document.getElementById('copyTitle').addEventListener('click', async () => {
  const tab = await getCurrentTab();
  copyToClipboard(tab.title, 'copyTitle');
});

// Copy as Markdown Link [title](url)
document.getElementById('copyMarkdownLink').addEventListener('click', async () => {
  const tab = await getCurrentTab();
  const markdown = `[${tab.title}](${tab.url})`;
  copyToClipboard(markdown, 'copyMarkdownLink');
});

// Copy as HTML Link <a href="url">title</a>
document.getElementById('copyHtmlLink').addEventListener('click', async () => {
  const tab = await getCurrentTab();
  const html = `<a href="${tab.url}">${tab.title}</a>`;
  copyToClipboard(html, 'copyHtmlLink');
});

// Copy Title + URL (two lines)
document.getElementById('copyTitleAndUrl').addEventListener('click', async () => {
  const tab = await getCurrentTab();
  const combined = `${tab.title}\n${tab.url}`;
  copyToClipboard(combined, 'copyTitleAndUrl');
});

// === PREMIUM FEATURES ===

// Copy all tabs as Markdown list
document.getElementById('copyAllTabs').addEventListener('click', async () => {
  if (!isPremium) {
    showUpgradePrompt();
    return;
  }
  const tabs = await chrome.tabs.query({ currentWindow: true });
  const markdown = tabs.map(tab => `- [${tab.title}](${tab.url})`).join('\n');
  copyToClipboard(markdown, 'copyAllTabs');
});

// View copy history
document.getElementById('viewHistory').addEventListener('click', async () => {
  if (!isPremium) {
    showUpgradePrompt();
    return;
  }
  chrome.storage.local.get(['copyHistory'], (result) => {
    const history = result.copyHistory || [];
    if (history.length === 0) {
      alert('No copy history yet!');
      return;
    }
    // Show last 5 items
    const preview = history.slice(0, 5).map((item, i) =>
      `${i + 1}. ${item.text.substring(0, 50)}...`
    ).join('\n');
    alert('Recent copies:\n\n' + preview);
  });
});
